<?php
/**
 * TinyPNG API handler for Image Compression Plugin
 */

if (!defined('ABSPATH')) {
    exit;
}

class ICP_TinyPNG_API {
    
    private $api_key;
    private $api_url = 'https://api.tinify.com/shrink';
    
    public function __construct() {
        $settings = get_option('icp_settings');
        $this->api_key = isset($settings['api_key']) ? $settings['api_key'] : '';
    }
    
    public function set_api_key($api_key) {
        $this->api_key = $api_key;
    }
    
    public function compress_image($image_path) {
        if (empty($this->api_key)) {
            return array(
                'success' => false,
                'error' => 'TinyPNG API key not configured'
            );
        }
        
        if (!file_exists($image_path)) {
            return array(
                'success' => false,
                'error' => 'Image file not found'
            );
        }
        
        $original_size = filesize($image_path);
        
        // Check if file is too large (TinyPNG limit is 5MB)
        if ($original_size > 5 * 1024 * 1024) {
            return array(
                'success' => false,
                'error' => 'Image file too large (max 5MB)'
            );
        }
        
        $image_data = file_get_contents($image_path);
        
        // Send to TinyPNG API
        $response = $this->send_to_tinypng($image_data);
        
        if (!$response['success']) {
            return $response;
        }
        
        // Download compressed image
        $compressed_data = $this->download_compressed_image($response['download_url']);
        
        if (!$compressed_data['success']) {
            return $compressed_data;
        }
        
        // Save compressed image - create backup first
        $backup_path = $image_path . '.icp_backup';
        copy($image_path, $backup_path);
        
        $result = file_put_contents($image_path, $compressed_data['data']);
        
        if ($result === false) {
            // Restore from backup if save failed
            copy($backup_path, $image_path);
            unlink($backup_path);
            return array(
                'success' => false,
                'error' => 'Failed to save compressed image'
            );
        }
        
        // Clear file stat cache to get accurate file size
        clearstatcache(true, $image_path);
        $compressed_size = filesize($image_path);
        
        // Clean up backup
        unlink($backup_path);
        
        if ($compressed_size === false || $compressed_size <= 0) {
            return array(
                'success' => false,
                'error' => 'Failed to get compressed file size'
            );
        }
        
        $savings_percent = $original_size > 0 ? (($original_size - $compressed_size) / $original_size) * 100 : 0;
        
        return array(
            'success' => true,
            'original_size' => $original_size,
            'compressed_size' => $compressed_size,
            'savings_percent' => round($savings_percent, 2),
            'savings_bytes' => $original_size - $compressed_size
        );
    }
    
    private function send_to_tinypng($image_data) {
        $args = array(
            'timeout' => 30,
            'headers' => array(
                'Authorization' => 'Basic ' . base64_encode('api:' . $this->api_key),
                'Content-Type' => 'application/octet-stream'
            ),
            'body' => $image_data
        );
        
        $response = wp_remote_post($this->api_url, $args);
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error' => 'API request failed: ' . $response->get_error_message()
            );
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        if ($response_code !== 201) {
            $error_data = json_decode($response_body, true);
            $error_message = isset($error_data['error']) ? $error_data['error'] : 'Unknown API error';
            
            return array(
                'success' => false,
                'error' => 'TinyPNG API error: ' . $error_message
            );
        }
        
        $data = json_decode($response_body, true);
        
        if (!isset($data['output']['url'])) {
            return array(
                'success' => false,
                'error' => 'Invalid API response'
            );
        }
        
        return array(
            'success' => true,
            'download_url' => $data['output']['url']
        );
    }
    
    private function download_compressed_image($download_url) {
        $args = array(
            'timeout' => 30,
            'headers' => array(
                'Authorization' => 'Basic ' . base64_encode('api:' . $this->api_key)
            )
        );
        
        $response = wp_remote_get($download_url, $args);
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error' => 'Failed to download compressed image: ' . $response->get_error_message()
            );
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        
        if ($response_code !== 200) {
            return array(
                'success' => false,
                'error' => 'Failed to download compressed image'
            );
        }
        
        return array(
            'success' => true,
            'data' => wp_remote_retrieve_body($response)
        );
    }
    
    public function test_api_key($api_key = null) {
        $test_key = $api_key ? $api_key : $this->api_key;
        
        if (empty($test_key)) {
            return array(
                'success' => false,
                'error' => 'API key is required'
            );
        }
        
        // Create a small test image (1x1 PNG)
        $test_image = base64_decode('iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNkYPhfDwAChwGA60e6kgAAAABJRU5ErkJggg==');
        
        $args = array(
            'timeout' => 15,
            'headers' => array(
                'Authorization' => 'Basic ' . base64_encode('api:' . $test_key),
                'Content-Type' => 'application/octet-stream'
            ),
            'body' => $test_image
        );
        
        $response = wp_remote_post($this->api_url, $args);
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error' => 'API connection failed: ' . $response->get_error_message()
            );
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        
        if ($response_code === 201) {
            return array(
                'success' => true,
                'message' => 'API key is valid'
            );
        } else if ($response_code === 401) {
            return array(
                'success' => false,
                'error' => 'Invalid API key'
            );
        } else {
            $response_body = wp_remote_retrieve_body($response);
            $error_data = json_decode($response_body, true);
            $error_message = isset($error_data['error']) ? $error_data['error'] : 'Unknown error';
            
            return array(
                'success' => false,
                'error' => 'API error: ' . $error_message
            );
        }
    }
    
    /**
     * Check if API key is configured and not empty
     */
    public function is_api_key_configured() {
        return !empty($this->api_key);
    }
    
    /**
     * Get the current API key
     */
    public function get_api_key() {
        return $this->api_key;
    }
} 